/*
 *  Copyright (C) 2010 Ryszard Wiśniewski <brut.alll@gmail.com>
 *  Copyright (C) 2010 Connor Tumbleson <connor.tumbleson@gmail.com>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       https://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package brut.androlib.util;

import brut.androlib.BaseTest;
import brut.util.BrutIO;
import brut.util.OSDetection;

import java.io.File;
import java.nio.file.InvalidPathException;

import org.junit.*;
import static org.junit.Assert.*;

public class InvalidDirectoryTraversalTest extends BaseTest {

    @BeforeClass
    public static void beforeClass() throws Exception {
        copyResourceDir(InvalidDirectoryTraversalTest.class, "util/traversal", sTmpDir);
    }

    @Test
    public void validFileTest() throws Exception {
        String validFileName = BrutIO.sanitizePath(sTmpDir, "file");
        assertEquals(validFileName, "file");
        assertTrue(new File(sTmpDir, validFileName).isFile());
    }

    @Test(expected = InvalidPathException.class)
    public void invalidBackwardFileTest() throws Exception {
        BrutIO.sanitizePath(sTmpDir, "../file");
    }

    @Test(expected = InvalidPathException.class)
    public void invalidRootFileTest() throws Exception {
        String rootLocation = OSDetection.isWindows() ? "C:/" : File.separator;
        BrutIO.sanitizePath(sTmpDir, rootLocation + "file");
    }

    @Test(expected = InvalidPathException.class)
    public void noFilePassedTest() throws Exception {
        BrutIO.sanitizePath(sTmpDir, "");
    }

    @Test(expected = InvalidPathException.class)
    public void invalidBackwardPathOnWindows() throws Exception {
        String invalidPath = OSDetection.isWindows() ? "..\\..\\app.exe" : "../../app";
        BrutIO.sanitizePath(sTmpDir, invalidPath);
    }

    @Test
    public void validDirectoryFileTest() throws Exception {
        String fileName = "dir" + File.separator + "file";
        String validFileName = BrutIO.sanitizePath(sTmpDir, fileName);
        assertEquals(fileName, validFileName);
    }
}
