// -*- mode: c++; indent-tabs-mode: nil -*-
// @file SalesforceSoapClient.qm Salesforce SOAP Client Definition Module

/*  SalesforceSoapClient.qm Copyright (C) 2015 - 2023 Qore Technologies, s.r.o.

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
*/





/** @mainpage SalesforceSoapClient Module

    @tableofcontents

    @section salesforcesoapclientintro Introduction to the SalesforceSoapClient Module

    The SalesforceSoapClient module provides an API for communicating with Salesforce.com using SOAP.

    This module provides the following classes:
    - @ref SalesforceSoapClient::SalesforceSoapClient "SalesforceSoapClient"
    - @ref SalesforceSoapClient::SalesforceSoapConnection "SalesforceSoapConnection"

    This module requires the following modules which are also reexported to the program context:
    - <a href="../../xml/html/index.html">xml binary module</a>
    - <a href="../../SoapClient/html/index.html">SoapClient user module</a>
    - <a href="../../WSDL/html/index.html">WSDL user module</a>
    - @ref mimeintro "Mime user module"

    @section salessoapclient_example SalesforceSoapClient Example

    Here is an example of how to use this module:
    @code{.py}
hash opts = (
    "wsdl": wsdl,
    "username": "myuser@example.com",
    "password": "pass",
    "token": "token",
    "log": sub (string fmt) {
        if (m_options.verbose)
            printf("INFO: %s\n", vsprintf(fmt, argv));
    },
    "dbglog": sub (string fmt) {
        if (m_options.verbose > 1)
            printf("DEBUG: %s\n", vsprintf(fmt, argv));
    },
    );

SalesforceSoapClient sc(opts);
string ss = sprintf("select id, name, description, accountnumber from account where accountnumber = '%s'", AcctNo1);
hash<auto> rh = sc.query({"queryString": ss});
printf("%N\n", rh);
    @endcode

    @section salesforcesoapclientrelnotes SalesforceSoapClient Release Notes

    @subsection salesfocesoapclient_1_4 SalesforceSoapClient v1.4
    - implemented support for a data provider scheme cache and rich option information for connections
      (<a href="https://github.com/qorelanguage/qore/issues/4025">issue 4025</a>)

    @subsection salesfocesoapclient_1_3 SalesforceSoapClient v1.3
    - removed the obsolete \c SalesforceSoapConnection::getConstructorInfo() method as connection serialization is a
      much more elegant and maintainable solution
      (<a href="https://github.com/qorelanguage/qore/issues/3696">issue 3696</a>)

    @subsection salesfocesoapclient_1_2 SalesforceSoapClient v1.2
    - added the \c SalesforceSoapConnection::getConstructorInfo()
      method to allow connections to be created dynamically, potentially in another process from a network
      call (<a href="https://github.com/qorelanguage/qore/issues/2628">issue 2628</a>)

    @subsection salesfocesoapclient_1_1 SalesforceSoapClient v1.1
    - added the @ref SalesforceSoapClient::SalesforceSoapConnection "SalesforceSoapConnection" class

    @subsection salesfocesoapclient_1_0 SalesforceSoapClient v1.0
    - initial release of the module
*/

//! main %SalesforceSoapClient module namespace
namespace SalesforceSoapClient {
//! class provising SOAP API access to Salesforce.com
/** This class inherits @ref SoapClient::SoapClient "SoapClient" to perform SOAP communication with
    Salesforce.com.

    @note By default this class does not log out of Salesforce.com in the destructor in order to keep the API session alive.
    To force a logout of the session in the destructor, use the \c force_logout option in the constructor().
*/
class SalesforceSoapClient : public SoapClient::SoapClient {

public:
protected:
        // mutex for atomic operations
        Mutex lck();
        // logged in flag
        bool logged_in = False;

        // Salesforce.com username
        string username;
        // Salesforce.com user password
        string password;
        // Salesforce.com user token
        string token;

        // Salesforce.com server URL
        string serverurl;
        // Salesforce.com session ID
        string sessionid;

        // default SoapAction value
        string soapaction = "default";

        // perform a logout when destroying the object
        /** logouts should be performed manually to ensure that the user's session remains valid
        */
        bool force_logout = False;

public:

        //! authorization key required for logging in to Salesforce.com
        const AuthorizationHash = ...;


        //! list of authorization keys
        const AuthorizationOpts = AuthorizationHash.keys();

        //! required options
        const RequiredOpts = ...;


    //! creates the object with the given options
    /** @param opts options for the SOAP connection; see @ref SoapClient::SoapClient::constructor() "SoapClient::constructor()" for option info for the embedded SoapClient, plus the following options:
        - \c username: (required) Salesforce.com username
        - \c password: (required) Salesforce.com password
        - \c token: (required) Salesforce.com user API token
        - \c force_logout (optional) forces a logout when the SalesforceSoapClient object is destroyed; this should normally be @ref Qore::False "False" to allow for the session to remain valid
    */
    constructor(hash<auto> opts) ;


    //! destroys the object
    /** @note if the \c force_logout option was given in the constructor(), then a logout from the Salesforce.com session is also performed
    */
    destructor();


    //! processes options given in the constructor()
protected:
     processOpts(hash<auto> opts);
public:


    //! performs a manual login to Salesforce.com
    login();


    //! performs a manual logout to Salesforce.com
    logout();


    //! disconnects from Salesforce.com; if the \c force_logout option was given in the constructor(), then a logout is performed before disconnecting
    disconnect();


    //! makes a server call with the given operation, arguments, options, and optional info hash reference and returns the result
    /** @param operation the SOAP operation to use to serialize the request; if the operation is not known to the underlying @ref WSDL::WebService "WebService" class, an exception will be thrown
        @param args the arguments to the SOAP operation
        @param opts an optional hash of options for the call as follows:
        - \c soap_header: a hash giving SOAP header information, if required by the message
        - \c http_header: a hash giving HTTP header information to include in the message (does not override automatically-generated SOAP message headers)
        - \c xml_opts: an integer XML generation option code; see @ref xml_generation_constants for possible values; combine multiple codes with binary or (\c |)
        - \c soapaction: an optional string that will override the SOAPAction for the request; en empty string here will prevent the SOAPAction from being sent
        @param info an optional reference to return a hash of technical information about the SOAP call (raw message info and headers); the following keys are present in this hash:
        - \c "headers": a hash of HTTP request headers
        - \c "request-uri": the request URI string (ex: \c "POST /services/Soap/c/29.0 HTTP/1.1")
        - \c "response-uri": the response URI string (ex: \c "HTTP/1.1 200 OK")
        - \c "charset": the character encoding string (ex: \c "UTF-8")
        - \c "body-content-type": the \c Content-Type of the response without any \c charset declaration
        - \c "accept-charset": the valus of any \c Accept-Charset header in the response
        - \c "response-headers": a hash of HTTP response headers
        - \c "response-body": the raw XML response body (in case content encoding is used, this is the decoded value)
        - \c "request-body": the raw XML request body
        - \c "request-soap-headers": an optional hash of SOAP headers used in the request (if applicable)

        @return a hash with the following keys:
        - \c hdr: a hash of message headers
        - \c body: the serialized message body

        @throw WSDL-OPERATION-ERROR the operation is not defined in the WSDL
        @throw HTTP-CLIENT-RECEIVE-ERROR this exception is thrown when the SOAP server returns an HTTP error code; if a SOAP fault is returned, then it is deserialized and returned in the \a arg key of the exception hash

        @note this method can throw any exception that @ref Qore::HTTPClient::send() "HTTPClient::send()" can throw as well as any XML parsing errors thrown by @ref Qore::XML::parse_xml() "parse_xml()"
    */
    auto callOperation(string operation, auto args, __7_ hash<auto> opts, __7_ reference<auto> info);


    //! makes the internal call and returns the deserialized response
    /** @param operation the SOAP operation to use to serialize the request; if the operation is not known to the underlying @ref WSDL::WebService "WebService" class, an exception will be thrown
        @param args the arguments to the SOAP operation
        @param opts an optional hash of options for the call as follows:
        - \c soap_header: a hash giving SOAP header information, if required by the message
        - \c http_header: a hash giving HTTP header information to include in the message (does not override automatically-generated SOAP message headers)
        - \c xml_opts: an integer XML generation option code; see @ref xml_generation_constants for possible values; combine multiple codes with binary or (\c |)
        - \c soapaction: an optional string that will override the SOAPAction for the request; en empty string here will prevent the SOAPAction from being sent
        @param info an optional reference to return a hash of technical information about the SOAP call (raw message info and headers); the following keys are present in this hash:
        - \c "headers": a hash of HTTP request headers
        - \c "request-uri": the request URI string (ex: \c "POST /services/Soap/c/29.0 HTTP/1.1")
        - \c "response-uri": the response URI string (ex: \c "HTTP/1.1 200 OK")
        - \c "charset": the character encoding string (ex: \c "UTF-8")
        - \c "body-content-type": the \c Content-Type of the response without any \c charset declaration
        - \c "accept-charset": the valus of any \c Accept-Charset header in the response
        - \c "response-headers": a hash of HTTP response headers
        - \c "response-body": the raw XML response body (in case content encoding is used, this is the decoded value)
        - \c "request-body": the raw XML request body
        - \c "request-soap-headers": an optional hash of SOAP headers used in the request (if applicable)

        @return a hash with the following keys:
        - \c hdr: a hash of message headers
        - \c body: the serialized message body

        @throw WSDL-OPERATION-ERROR the operation is not defined in the WSDL
        @throw HTTP-CLIENT-RECEIVE-ERROR this exception is thrown when the SOAP server returns an HTTP error code; if a SOAP fault is returned, then it is deserialized and returned in the \a arg key of the exception hash

        @note this method can throw any exception that @ref Qore::HTTPClient::send() "HTTPClient::send()" can throw as well as any XML parsing errors thrown by @ref Qore::XML::parse_xml() "parse_xml()"
    */
protected:
     auto callIntern(string operation, auto args, __7_ hash<auto> opts, __7_ reference<auto> info);
public:


    //! makes a server call with the given operation and arguments and returns the deserialized result
    /** @param operation the operation name for the SOAP call
        @param args the operation parameter(s)
        @param header optional soap headers (if required by the operation)

        @return the deserialized result of the SOAP call to the SOAP server
    */
    auto call(string operation, auto args, __7_ hash<auto> header);


    //! makes a server call with the given operation and arguments and returns the deserialized result
    /** @param operation the operation name for the SOAP call
        @param args the arguments to the SOAP operation
        @param info an optional reference to return a hash of technical information about the SOAP call (raw message info and headers); the following keys are present in this hash:
        - \c "headers": a hash of HTTP request headers
        - \c "request-uri": the request URI string (ex: \c "POST /services/Soap/c/29.0 HTTP/1.1")
        - \c "response-uri": the response URI string (ex: \c "HTTP/1.1 200 OK")
        - \c "charset": the character encoding string (ex: \c "UTF-8")
        - \c "body-content-type": the \c Content-Type of the response without any \c charset declaration
        - \c "accept-charset": the valus of any \c Accept-Charset header in the response
        - \c "response-headers": a hash of HTTP response headers
        - \c "response-body": the raw XML response body (in case content encoding is used, this is the decoded value)
        - \c "request-body": the raw XML request body
        - \c "request-soap-headers": an optional hash of SOAP headers used in the request (if applicable)

        @return the deserialized result of the SOAP call to the SOAP server
    */
    auto call(string operation, auto args, reference<auto> info);


    //! makes a server call with the given operation and arguments and returns the deserialized result with an output argument giving technical information about the call
    /** @param info an optional reference to return a hash of technical information about the SOAP call (raw message info and headers); the following keys are present in this hash:
        - \c "headers": a hash of HTTP request headers
        - \c "request-uri": the request URI string (ex: \c "POST /services/Soap/c/29.0 HTTP/1.1")
        - \c "response-uri": the response URI string (ex: \c "HTTP/1.1 200 OK")
        - \c "charset": the character encoding string (ex: \c "UTF-8")
        - \c "body-content-type": the \c Content-Type of the response without any \c charset declaration
        - \c "accept-charset": the valus of any \c Accept-Charset header in the response
        - \c "response-headers": a hash of HTTP response headers
        - \c "response-body": the raw XML response body (in case content encoding is used, this is the decoded value)
        - \c "request-body": the raw XML request body
        - \c "request-soap-headers": an optional hash of SOAP headers used in the request (if applicable)
        @param operation the operation name for the SOAP call
        @param args the arguments to the SOAP operation
        @param header optional soap headers (if required by the operation)

        @return the deserialized result of the SOAP call to the SOAP server
    */
    auto call(reference<auto> info, string operation, auto args, __7_ hash<auto> header);


    nothing setWarningQueue(int warning_ms, int warning_bs, Queue queue, auto arg, timeout min_ms = 1s);


    nothing clearWarningQueue();


    //! uses call() to transparently serialize the argument and make a call to the given operation and return the deserialized results
    /** @param op the operation name, which is the method name passed to methodGate()
        @param arg a list or arguments or a single argument (or NOTHING) for the operation

        @return the deserialized result of the SOAP call to the SOAP server
    */
    auto methodGate(string op, auto arg);


protected:
     logoutIntern();
public:


    // must be called with the lock held
protected:
     loginIntern();
public:

};

//! class for SOAP connections to the Salesforce.com SOAP API; returns an object of class @ref SalesforceSoapClient
/** supports all @ref SoapClient::SoapConnection "SoapConnection" options plus the following static initialization
    options:
    - \c "force_logout": forces a logout when the SalesforceSoapClient object is destroyed; this should normally be
      @ref Qore::False "False" to allow for the session to remain valid
    - \c "password": Salesforce.com password
    - \c "token": Salesforce.com user API token
    - \c "username": Salesforce.com username

    @see @ref SalesforceSoapClient::constructor(hash) for more information on the above options

    @note
    - additionally supports the following runtime options in getImpl() for connection logging:
        - \c "log": a closure accepting a single string for logging
        - \c "dbglog": a closure taking a single string for detailed technical connection logging
    - the following options are required:
        - \c "username": (required) Salesforce.com username
        - \c "password": (required) Salesforce.com password
        - \c "token": (required) Salesforce.com user API token
*/
class SalesforceSoapConnection : public SoapConnection {

public:
        //! Connection entry info
        const ConnectionScheme = ...;


protected:
        //! cache of WebService objects keyed by file location
        hash cache;

public:

    //! creates the SalesforceSoapConnection object
    /** @param name the name of the connection
        @param description connection description
        @param url connection URL (potentially with password info)
        @param attributes various attributes. See below
        @param options connection options

        See @ref AbstractConnection::constructor() for \c attributes and \c options reference.

        @throw SALESFORCE-SOAP-ERROR missing one or more of the required options: \c "username", \c "password", or \c "token"
    */
    constructor(string name, string description, string url, hash<auto> attributes = {}, hash<auto> options = {})
 ;


    //! Creates the SalesforceSoapConnection object
    /** @param config with the following keys:
        - name (required string): the connection name
        - display_name (optional string): the display name
        - short_desc (optional string): a short description in plain text
        - desc (optional string): a long description with markdown formatting
        - url (required string): the connection URL
        - opts (optional hash): connection options
        - logger (optional LoggerInterface object): logger for the connection
        @param attr optional connection attributes
        - monitor (optional bool): should the connection be monitored? Default: True
        - enabled (optional bool): is the connection enabled? Default: True
        - locked (optional bool): is the connection locked? Default: False
        - debug_data (optional bool): debug data? Default: False
        - tags (optional hash): tags for the connection; no default value

        @throw CONNECTION-OPTION-ERROR missing or invalid connection option or attribute
    */
    constructor(hash<auto> config, __7_ hash<auto> attr) ;


    //! Ensures that required options are set
protected:
     checkOpts();
public:


    //! returns \c "sfsoap"
    string getType();


    //! returns runtime options
    /** @return a hash with the following keys reflecting support for the corresponding runtime options in getImpl() for connection logging:
        - \c "log": a closure accepting a single string for logging
        - \c "dbglog": a closure taking a single string for detailed technical connection logging
    */
    __7_ hash<auto> getRuntimeOptions();


    //! returns @ref False
    /** This connection type has not yet been updated to support the %Qore @ref Qore::Socket "Socket-based" polling
        API

        @return @ref False
    */
    bool supportsPollingApi();


    //! returns a @ref SalesforceSoapClient object
    /** @param connect if @ref Qore::True "True", then @ref SalesforceSoapClient::login() is called
        @param rtopts an optional hash providing two runtime options for connection logging:
        - \c "log": a closure accepting a single string for logging
        - \c "dbglog": a closure taking a single string for detailed technical connection logging

        @return a @ref SalesforceSoapClient object
    */
protected:
     SalesforceSoapClient getImpl(bool connect = True, __7_ hash<auto> rtopts);
public:


    //! returns a WSDL::WebService object for the given URL using a cache for local files
    /** @param url the URL of the WSDL

        @return a WSDL::WebService object corresponding to the WSDL URL
        */
protected:
     WSDL::WebService getWsdl(string url);
public:


    //! returns a WSDL::WebService object for the given path from the cache if possible, otherwise creates the object and populates the cache
protected:
      synchronized WSDL::WebService getWsdlFromCache(string path);
public:


    //! returns a WSDL::WebService object from the given URL
protected:
     WSDL::WebService getWebService(string url);
public:


    //! Returns the ConnectionSchemeInfo hash for this object
protected:
     hash<ConnectionSchemeInfo> getConnectionSchemeInfoImpl();
public:

};
};
