import { PartialJSONObject } from '@lumino/coreutils';
/**
 * The namespace for URL-related functions.
 */
export declare namespace URLExt {
    /**
     * Parse a url into a URL object.
     *
     * @param urlString - The URL string to parse.
     *
     * @returns A URL object.
     */
    function parse(url: string): IUrl;
    /**
     * Parse URL and retrieve hostname
     *
     * @param url - The URL string to parse
     *
     * @returns a hostname string value
     */
    function getHostName(url: string): string;
    /**
     * Normalize a url.
     */
    function normalize(url: string): string;
    function normalize(url: undefined): undefined;
    function normalize(url: string | undefined): string | undefined;
    /**
     * Join a sequence of url components and normalizes as in node `path.join`.
     *
     * @param parts - The url components.
     *
     * @returns the joined url.
     */
    function join(...parts: string[]): string;
    /**
     * Encode the components of a multi-segment url.
     *
     * @param url - The url to encode.
     *
     * @returns the encoded url.
     *
     * #### Notes
     * Preserves the `'/'` separators.
     * Should not include the base url, since all parts are escaped.
     */
    function encodeParts(url: string): string;
    /**
     * Return a serialized object string suitable for a query.
     *
     * @param object - The source object.
     *
     * @returns an encoded url query.
     *
     * #### Notes
     * Modified version of [stackoverflow](http://stackoverflow.com/a/30707423).
     */
    function objectToQueryString(value: PartialJSONObject): string;
    /**
     * Return a parsed object that represents the values in a query string.
     */
    function queryStringToObject(value: string): {
        [key: string]: string | undefined;
    };
    /**
     * Test whether the url is a local url.
     *
     * #### Notes
     * This function returns `false` for any fully qualified url, including
     * `data:`, `file:`, and `//` protocol URLs.
     */
    function isLocal(url: string): boolean;
    /**
     * The interface for a URL object
     */
    interface IUrl {
        /**
         * The full URL string that was parsed with both the protocol and host
         * components converted to lower-case.
         */
        href: string;
        /**
         * Identifies the URL's lower-cased protocol scheme.
         */
        protocol: string;
        /**
         * The full lower-cased host portion of the URL, including the port if
         * specified.
         */
        host: string;
        /**
         * The lower-cased host name portion of the host component without the
         * port included.
         */
        hostname: string;
        /**
         * The numeric port portion of the host component.
         */
        port: string;
        /**
         * The entire path section of the URL.
         */
        pathname: string;
        /**
         * The "fragment" portion of the URL including the leading ASCII hash
         * `(#)` character
         */
        hash: string;
        /**
         * The search element, including leading question mark (`'?'`), if any,
         * of the URL.
         */
        search?: string;
    }
}
